<?php
/**
 * Paystack Payment Gateway Integration
 * Supports: Nigeria, Ghana, South Africa
 * API Documentation: https://paystack.com/docs/api
 */

class Paystack {
    private $secretKey;
    private $publicKey;
    private $apiUrl = 'https://api.paystack.co';
    private $db;
    
    public function __construct($config) {
        $this->secretKey = $config['secret_key'] ?? '';
        $this->publicKey = $config['public_key'] ?? '';
        $this->db = Database::getInstance();
    }
    
    /**
     * Initialize payment
     */
    public function initializePayment($data) {
        $reference = 'PS_' . time() . '_' . bin2hex(random_bytes(8));
        
        // Convert amount to kobo/pesewas (multiply by 100)
        $amountInMinorUnits = $data['amount'] * 100;
        
        $payload = [
            'reference' => $reference,
            'amount' => $amountInMinorUnits,
            'currency' => $data['currency'] ?? 'NGN',
            'email' => $data['customer_email'],
            'callback_url' => $data['callback_url'],
            'metadata' => [
                'user_id' => $data['user_id'],
                'customer_name' => $data['customer_name'],
                'transaction_id' => $data['transaction_id'] ?? null,
                'auto_topup' => $data['auto_topup'] ?? false,
                'auto_topup_session' => $data['auto_topup_session'] ?? null
            ]
        ];
        
        $response = $this->makeRequest('/transaction/initialize', $payload);
        
        if ($response['success']) {
            $this->logPayment('info', 'Payment initialized', [
                'reference' => $reference,
                'amount' => $data['amount']
            ], $data['user_id']);
            
            return [
                'success' => true,
                'reference' => $reference,
                'checkout_url' => $response['data']['authorization_url'] ?? null,
                'access_code' => $response['data']['access_code'] ?? null
            ];
        }
        
        $this->logPayment('error', 'Initialization failed', $response, $data['user_id']);
        return ['success' => false, 'message' => $response['message'] ?? 'Failed'];
    }
    
    /**
     * Verify payment
     */
    public function verifyPayment($reference) {
        $response = $this->makeRequest("/transaction/verify/{$reference}", null, 'GET');
        
        if ($response['success'] && $response['data']['status'] === 'success') {
            return [
                'success' => true,
                'status' => 'success',
                'amount' => $response['data']['amount'] / 100, // Convert back from kobo
                'currency' => $response['data']['currency'] ?? 'NGN',
                'reference' => $reference,
                'paid_at' => $response['data']['paid_at'] ?? null,
                'metadata' => $response['data']['metadata'] ?? []
            ];
        }
        
        return ['success' => false, 'message' => 'Verification failed'];
    }
    
    /**
     * Process webhook
     */
    public function processWebhook($payload) {
        // Verify webhook signature
        $signature = $_SERVER['HTTP_X_PAYSTACK_SIGNATURE'] ?? '';
        $computed = hash_hmac('sha512', json_encode($payload), $this->secretKey);
        
        if (!hash_equals($computed, $signature)) {
            $this->logPayment('error', 'Invalid webhook signature');
            return ['success' => false, 'message' => 'Invalid signature'];
        }
        
        $event = $payload['event'] ?? '';
        
        if ($event === 'charge.success') {
            return $this->handleSuccessfulPayment($payload['data']);
        }
        
        return ['success' => true];
    }
    
    /**
     * Handle successful payment
     */
    private function handleSuccessfulPayment($data) {
        $reference = $data['reference'] ?? '';
        $amount = ($data['amount'] ?? 0) / 100; // Convert from kobo
        
        $this->db->query("SELECT * FROM transactions WHERE payment_reference = :ref")
                 ->bind(':ref', $reference);
        $transaction = $this->db->getOne();
        
        if (!$transaction || $transaction['status'] === 'completed') {
            return ['success' => true];
        }
        
        $this->db->beginTransaction();
        
        try {
            $this->db->update('transactions', [
                'status' => 'completed',
                'updated_at' => date('Y-m-d H:i:s')
            ], ['id' => $transaction['id']]);
            
            $this->db->query("UPDATE users SET balance = balance + :amount WHERE id = :user_id")
                     ->bind(':amount', $transaction['net_amount'])
                     ->bind(':user_id', $transaction['user_id'])
                     ->execute();
            
            $this->db->insert('user_activity_logs', [
                'user_id' => $transaction['user_id'],
                'action' => 'deposit_completed',
                'description' => "Deposit of {$transaction['net_amount']} completed via Paystack",
                'ip_address' => Security::getClientIP()
            ]);
            
            $this->db->commit();
            
            $this->logPayment('success', 'Payment processed', [
                'reference' => $reference,
                'amount' => $amount
            ], $transaction['user_id']);
            
            return ['success' => true];
            
        } catch (Exception $e) {
            $this->db->rollback();
            $this->logPayment('error', 'Processing failed: ' . $e->getMessage());
            return ['success' => false];
        }
    }
    
    /**
     * Make API request
     */
    private function makeRequest($endpoint, $data = null, $method = 'POST') {
        $url = $this->apiUrl . $endpoint;
        
        $headers = [
            'Authorization: Bearer ' . $this->secretKey,
            'Content-Type: application/json',
            'Cache-Control: no-cache'
        ];
        
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, true);
        curl_setopt($ch, CURLOPT_TIMEOUT, 30);
        
        if ($method === 'POST' && $data) {
            curl_setopt($ch, CURLOPT_POST, true);
            curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
        }
        
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        
        $result = json_decode($response, true);
        
        if ($httpCode >= 200 && $httpCode < 300 && ($result['status'] ?? false)) {
            return ['success' => true, 'data' => $result['data'] ?? []];
        }
        
        return ['success' => false, 'message' => $result['message'] ?? 'Failed'];
    }
    
    /**
     * Log payment activity
     */
    private function logPayment($type, $message, $data = [], $userId = null) {
        $this->db->insert('payment_logs', [
            'user_id' => $userId,
            'payment_method' => 'paystack',
            'log_type' => $type,
            'message' => $message,
            'response_data' => json_encode($data),
            'ip_address' => Security::getClientIP()
        ]);
    }
}
