<?php
/**
 * Korapay Payment Gateway Integration
 * Supports: Nigeria (NGN), USD
 * API Documentation: https://korapay.com/docs
 */

class Korapay {
    private $publicKey;
    private $secretKey;
    private $apiUrl = 'https://api.korapay.com/merchant/api/v1';
    private $db;
    
    public function __construct($config) {
        $this->publicKey = $config['public_key'] ?? '';
        $this->secretKey = $config['secret_key'] ?? '';
        $this->db = Database::getInstance();
    }
    
    /**
     * Initialize payment
     */
    public function initializePayment($data) {
        $reference = 'KRP_' . time() . '_' . bin2hex(random_bytes(8));
        
        $payload = [
            'reference' => $reference,
            'amount' => $data['amount'],
            'currency' => $data['currency'] ?? 'NGN',
            'customer' => [
                'name' => $data['customer_name'],
                'email' => $data['customer_email']
            ],
            'redirect_url' => $data['callback_url'],
            'notification_url' => $data['webhook_url'] ?? $data['callback_url'],
            'merchant_bears_cost' => $data['merchant_bears_cost'] ?? false,
            'metadata' => [
                'user_id' => $data['user_id'],
                'transaction_id' => $data['transaction_id'] ?? null,
                'auto_topup' => $data['auto_topup'] ?? false,
                'auto_topup_session' => $data['auto_topup_session'] ?? null
            ]
        ];
        
        $response = $this->makeRequest('/charges/initialize', $payload, 'POST');
        
        if ($response['success']) {
            $this->logPayment('info', 'Payment initialized', [
                'reference' => $reference,
                'amount' => $data['amount'],
                'currency' => $data['currency'] ?? 'NGN'
            ], $data['user_id']);
            
            return [
                'success' => true,
                'reference' => $reference,
                'checkout_url' => $response['data']['checkout_url'] ?? null,
                'data' => $response['data']
            ];
        }
        
        $this->logPayment('error', 'Payment initialization failed', $response, $data['user_id']);
        
        return [
            'success' => false,
            'message' => $response['message'] ?? 'Payment initialization failed'
        ];
    }
    
    /**
     * Verify payment
     */
    public function verifyPayment($reference) {
        $response = $this->makeRequest("/charges/{$reference}", null, 'GET');
        
        if ($response['success'] && isset($response['data'])) {
            $paymentData = $response['data'];
            
            $this->logPayment('info', 'Payment verified', $paymentData);
            
            return [
                'success' => true,
                'status' => $paymentData['status'] ?? 'pending',
                'amount' => $paymentData['amount'] ?? 0,
                'currency' => $paymentData['currency'] ?? 'NGN',
                'reference' => $reference,
                'paid_at' => $paymentData['paid_at'] ?? null,
                'customer' => $paymentData['customer'] ?? [],
                'metadata' => $paymentData['metadata'] ?? []
            ];
        }
        
        return [
            'success' => false,
            'message' => $response['message'] ?? 'Payment verification failed'
        ];
    }
    
    /**
     * Process webhook
     */
    public function processWebhook($payload) {
        // Verify webhook signature if provided
        $signature = $_SERVER['HTTP_X_KORAPAY_SIGNATURE'] ?? '';
        
        if (!empty($signature) && !$this->verifyWebhookSignature($payload, $signature)) {
            $this->logPayment('error', 'Invalid webhook signature', ['signature' => $signature]);
            return ['success' => false, 'message' => 'Invalid signature'];
        }
        
        $event = $payload['event'] ?? '';
        $data = $payload['data'] ?? [];
        
        $this->logPayment('info', 'Webhook received', ['event' => $event, 'data' => $data]);
        
        if ($event === 'charge.success') {
            return $this->handleSuccessfulPayment($data);
        }
        
        return ['success' => true, 'message' => 'Webhook processed'];
    }
    
    /**
     * Handle successful payment
     */
    private function handleSuccessfulPayment($data) {
        $reference = $data['reference'] ?? '';
        $amount = $data['amount'] ?? 0;
        $metadata = $data['metadata'] ?? [];
        
        // Find transaction
        $this->db->query("SELECT * FROM transactions WHERE payment_reference = :ref")
                 ->bind(':ref', $reference);
        
        $transaction = $this->db->getOne();
        
        if (!$transaction) {
            $this->logPayment('error', 'Transaction not found', ['reference' => $reference]);
            return ['success' => false, 'message' => 'Transaction not found'];
        }
        
        if ($transaction['status'] === 'completed') {
            $this->logPayment('warning', 'Transaction already processed', ['reference' => $reference]);
            return ['success' => true, 'message' => 'Already processed'];
        }
        
        // Begin transaction
        $this->db->beginTransaction();
        
        try {
            // Update transaction status
            $this->db->update('transactions', [
                'status' => 'completed',
                'updated_at' => date('Y-m-d H:i:s')
            ], ['id' => $transaction['id']]);
            
            // Credit user balance
            $this->db->query("UPDATE users SET balance = balance + :amount WHERE id = :user_id")
                     ->bind(':amount', $transaction['net_amount'])
                     ->bind(':user_id', $transaction['user_id'])
                     ->execute();
            
            // Log activity
            $this->db->insert('user_activity_logs', [
                'user_id' => $transaction['user_id'],
                'action' => 'deposit_completed',
                'description' => "Deposit of {$transaction['net_amount']} {$transaction['metadata']->currency ?? 'USD'} completed via Korapay",
                'ip_address' => Security::getClientIP()
            ]);
            
            // Handle auto-topup if applicable
            if ($metadata['auto_topup'] ?? false) {
                $this->processAutoTopup($metadata['auto_topup_session'], $transaction);
            }
            
            $this->db->commit();
            
            $this->logPayment('success', 'Payment processed successfully', [
                'reference' => $reference,
                'amount' => $amount,
                'user_id' => $transaction['user_id']
            ], $transaction['user_id']);
            
            return ['success' => true, 'message' => 'Payment processed'];
            
        } catch (Exception $e) {
            $this->db->rollback();
            $this->logPayment('error', 'Payment processing failed', ['error' => $e->getMessage()], $transaction['user_id']);
            return ['success' => false, 'message' => 'Processing failed'];
        }
    }
    
    /**
     * Process auto-topup after successful payment
     */
    private function processAutoTopup($sessionToken, $transaction) {
        // Get auto-topup session
        $this->db->query("SELECT * FROM auto_topup_sessions WHERE session_token = :token AND status = 'payment_processing'")
                 ->bind(':token', $sessionToken);
        
        $session = $this->db->getOne();
        
        if (!$session) {
            return;
        }
        
        // Get current user balance
        $this->db->query("SELECT balance FROM users WHERE id = :user_id")
                 ->bind(':user_id', $session['user_id']);
        $user = $this->db->getOne();
        
        $currentBalance = $user['balance'] ?? 0;
        
        // Check if balance is sufficient now
        if ($currentBalance >= $session['order_amount']) {
            // Place the order
            require_once __DIR__ . '/../app/controllers/OrderController.php';
            $orderController = new OrderController();
            
            $orderResult = $orderController->placeOrder([
                'user_id' => $session['user_id'],
                'service_id' => $session['service_id'],
                'link' => $session['link'],
                'quantity' => $session['quantity'],
                'auto_topup' => true,
                'topup_transaction_id' => $transaction['id']
            ]);
            
            if ($orderResult['success']) {
                // Update session
                $this->db->update('auto_topup_sessions', [
                    'status' => 'completed',
                    'transaction_id' => $transaction['id'],
                    'completed_at' => date('Y-m-d H:i:s')
                ], ['id' => $session['id']]);
                
                $this->logPayment('success', 'Auto-topup order placed', [
                    'session_token' => $sessionToken,
                    'order_id' => $orderResult['order_id']
                ], $session['user_id']);
            }
        }
    }
    
    /**
     * Verify webhook signature
     */
    private function verifyWebhookSignature($payload, $signature) {
        $computed = hash_hmac('sha256', json_encode($payload), $this->secretKey);
        return hash_equals($computed, $signature);
    }
    
    /**
     * Make API request
     */
    private function makeRequest($endpoint, $data = null, $method = 'POST') {
        $url = $this->apiUrl . $endpoint;
        
        $headers = [
            'Authorization: Bearer ' . $this->secretKey,
            'Content-Type: application/json',
            'Accept: application/json'
        ];
        
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, true);
        curl_setopt($ch, CURLOPT_TIMEOUT, 30);
        
        if ($method === 'POST') {
            curl_setopt($ch, CURLOPT_POST, true);
            if ($data) {
                curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
            }
        }
        
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $error = curl_error($ch);
        curl_close($ch);
        
        if ($error) {
            return ['success' => false, 'message' => $error];
        }
        
        $result = json_decode($response, true);
        
        if ($httpCode >= 200 && $httpCode < 300) {
            return ['success' => true, 'data' => $result['data'] ?? $result];
        }
        
        return [
            'success' => false,
            'message' => $result['message'] ?? 'Request failed',
            'errors' => $result['errors'] ?? []
        ];
    }
    
    /**
     * Log payment activity
     */
    private function logPayment($type, $message, $data = [], $userId = null) {
        $this->db->insert('payment_logs', [
            'user_id' => $userId,
            'payment_method' => 'korapay',
            'log_type' => $type,
            'message' => $message,
            'response_data' => json_encode($data),
            'ip_address' => Security::getClientIP()
        ]);
    }
}
