<?php
/**
 * Flutterwave Payment Gateway Integration
 * Supports: Nigeria, Kenya, Ghana, South Africa, and more
 * API Documentation: https://developer.flutterwave.com
 */

class Flutterwave {
    private $publicKey;
    private $secretKey;
    private $encryptionKey;
    private $apiUrl = 'https://api.flutterwave.com/v3';
    private $db;
    
    public function __construct($config) {
        $this->publicKey = $config['public_key'] ?? '';
        $this->secretKey = $config['secret_key'] ?? '';
        $this->encryptionKey = $config['encryption_key'] ?? '';
        $this->db = Database::getInstance();
    }
    
    /**
     * Initialize payment
     */
    public function initializePayment($data) {
        $reference = 'FLW_' . time() . '_' . bin2hex(random_bytes(8));
        
        $payload = [
            'tx_ref' => $reference,
            'amount' => $data['amount'],
            'currency' => $data['currency'] ?? 'NGN',
            'redirect_url' => $data['callback_url'],
            'payment_options' => 'card,banktransfer,ussd,mobilemoney,mpesa',
            'customer' => [
                'email' => $data['customer_email'],
                'name' => $data['customer_name'],
                'phonenumber' => $data['customer_phone'] ?? ''
            ],
            'customizations' => [
                'title' => 'Add Funds',
                'description' => 'SMM Panel Balance Top-up',
                'logo' => $data['logo'] ?? ''
            ],
            'meta' => [
                'user_id' => $data['user_id'],
                'transaction_id' => $data['transaction_id'] ?? null,
                'auto_topup' => $data['auto_topup'] ?? false,
                'auto_topup_session' => $data['auto_topup_session'] ?? null
            ]
        ];
        
        $response = $this->makeRequest('/payments', $payload);
        
        if ($response['success']) {
            $this->logPayment('info', 'Payment initialized', [
                'reference' => $reference,
                'amount' => $data['amount']
            ], $data['user_id']);
            
            return [
                'success' => true,
                'reference' => $reference,
                'checkout_url' => $response['data']['link'] ?? null
            ];
        }
        
        $this->logPayment('error', 'Initialization failed', $response, $data['user_id']);
        return ['success' => false, 'message' => $response['message'] ?? 'Failed'];
    }
    
    /**
     * Verify payment
     */
    public function verifyPayment($reference) {
        $response = $this->makeRequest("/transactions/verify_by_reference?tx_ref={$reference}", null, 'GET');
        
        if ($response['success'] && $response['data']['status'] === 'successful') {
            return [
                'success' => true,
                'status' => 'success',
                'amount' => $response['data']['amount'] ?? 0,
                'currency' => $response['data']['currency'] ?? 'NGN',
                'reference' => $reference,
                'metadata' => $response['data']['meta'] ?? []
            ];
        }
        
        return ['success' => false, 'message' => 'Verification failed'];
    }
    
    /**
     * Process webhook
     */
    public function processWebhook($payload) {
        $signature = $_SERVER['HTTP_VERIF_HASH'] ?? '';
        
        if ($signature !== $this->secretKey) {
            $this->logPayment('error', 'Invalid webhook signature');
            return ['success' => false, 'message' => 'Invalid signature'];
        }
        
        if ($payload['status'] === 'successful') {
            return $this->handleSuccessfulPayment($payload);
        }
        
        return ['success' => true];
    }
    
    /**
     * Handle successful payment (similar to Korapay)
     */
    private function handleSuccessfulPayment($data) {
        $reference = $data['tx_ref'] ?? '';
        $amount = $data['amount'] ?? 0;
        
        $this->db->query("SELECT * FROM transactions WHERE payment_reference = :ref")
                 ->bind(':ref', $reference);
        $transaction = $this->db->getOne();
        
        if (!$transaction || $transaction['status'] === 'completed') {
            return ['success' => true];
        }
        
        $this->db->beginTransaction();
        
        try {
            $this->db->update('transactions', [
                'status' => 'completed',
                'updated_at' => date('Y-m-d H:i:s')
            ], ['id' => $transaction['id']]);
            
            $this->db->query("UPDATE users SET balance = balance + :amount WHERE id = :user_id")
                     ->bind(':amount', $transaction['net_amount'])
                     ->bind(':user_id', $transaction['user_id'])
                     ->execute();
            
            $this->db->commit();
            
            $this->logPayment('success', 'Payment processed', [
                'reference' => $reference,
                'amount' => $amount
            ], $transaction['user_id']);
            
            return ['success' => true];
            
        } catch (Exception $e) {
            $this->db->rollback();
            return ['success' => false];
        }
    }
    
    /**
     * Make API request
     */
    private function makeRequest($endpoint, $data = null, $method = 'POST') {
        $url = $this->apiUrl . $endpoint;
        
        $headers = [
            'Authorization: Bearer ' . $this->secretKey,
            'Content-Type: application/json'
        ];
        
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, true);
        curl_setopt($ch, CURLOPT_TIMEOUT, 30);
        
        if ($method === 'POST' && $data) {
            curl_setopt($ch, CURLOPT_POST, true);
            curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
        }
        
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        
        $result = json_decode($response, true);
        
        if ($httpCode >= 200 && $httpCode < 300 && ($result['status'] === 'success' || $result['status'] === 'successful')) {
            return ['success' => true, 'data' => $result['data'] ?? $result];
        }
        
        return ['success' => false, 'message' => $result['message'] ?? 'Failed'];
    }
    
    /**
     * Log payment activity
     */
    private function logPayment($type, $message, $data = [], $userId = null) {
        $this->db->insert('payment_logs', [
            'user_id' => $userId,
            'payment_method' => 'flutterwave',
            'log_type' => $type,
            'message' => $message,
            'response_data' => json_encode($data),
            'ip_address' => Security::getClientIP()
        ]);
    }
}
